﻿namespace Hims.Infrastructure.Services
{
    using System;
    using System.Threading.Tasks;
    using System.Collections.Generic;
    using Domain.Entities;
    using Domain.Repositories.UnitOfWork;
    using Domain.Services;
    using Shared.EntityModels;
    using Dapper;
    using Hims.Shared.UserModels.UserExcel;
    using Hims.Shared.UserModels.PatientExcel;

    /// <inheritdoc />
    public class PatientExcelHistoryService : IPatientExcelHistoryService
    {
        /// <summary>
        /// The unit of work.
        /// </summary>
        private readonly IUnitOfWork unitOfWork;

        /// <inheritdoc cref="IExcelUploadHistoryService" />
        public PatientExcelHistoryService(IUnitOfWork unitOfWork) => this.unitOfWork = unitOfWork;

        /// <inheritdoc />
        public async Task<IEnumerable<ExcelUploadHistoryModel>> FetchAllAsync(ExcelUploadHistoryModel model)
        {
            var where = "where 1=1 ";

            if (!string.IsNullOrEmpty(model.TypeOf))
            {
                where += $@" and EUH.""TypeOf"" ='{model.TypeOf}'";
            }

            var query = $@" Select count(EUH.*) over() as ""TotalItems"",EUH.*, A.""FullName"" as ""UploadedByName"", l.""Name"" as ""LocationName"" from ""ExcelUploadHistory"" EUH 
                            join ""Account"" A on A.""AccountId"" = EUH.""UploadedBy""
                            left join ""Location"" l on l.""LocationId"" = EUH.""LocationId""
                            {where}
                            order by ""CreatedDate"" desc";

            if (model.PageIndex != null && model.PageSize != null)
            {
                model.PageIndex = model.PageIndex > 0 ? model.PageIndex - 1 : model.PageIndex;
                query += $@" limit {model.PageSize} offset {model.PageSize * model.PageIndex}";
            }

            return await this.unitOfWork.Current.QueryAsync<ExcelUploadHistoryModel>(query);
        }

        /// <inheritdoc />
        public async Task<PatientExcelHistoryModel> GetSingleExcelHistory(int patientExcelHistoryId)
        {
            var data = await this.unitOfWork.PatientsExcelHistory.FindAsync(m => m.PatientExcelHistoryId == patientExcelHistoryId);
            return new PatientExcelHistoryModel
            {

                CreatedDate = data.CreatedDate,
                PatientExcelHistoryId = data.PatientExcelHistoryId,
                SheetName = data.SheetName,
                UploadedBy = data.UploadedBy
            };
        }

        public async Task<int> ModifyExcelUploadAsync(PatientExcelHistoryModel model)
        {
            var upload = new PatientExcelHistory
            {
                CreatedDate = DateTime.UtcNow.AddMinutes(330),
                SheetName = model.SheetName,
                UploadedBy = model.UploadedBy,
                LocationId = model.LocationId,
                AddedPatients = model.AddedPatients,
            };
            if(model.PatientExcelHistoryId == 0)
            {
                return await this.unitOfWork.PatientsExcelHistory.InsertAsync(upload);
            }
            else
            {
                var previousRecord = await this.unitOfWork.PatientsExcelHistory.FindAsync(m => m.PatientExcelHistoryId == model.PatientExcelHistoryId);
                var completeUserList = previousRecord.AddedPatients + "," + upload.AddedPatients;
                previousRecord.AddedPatients = completeUserList;
                await this.unitOfWork.PatientsExcelHistory.UpdateAsync(previousRecord);
                return previousRecord.PatientExcelHistoryId;
            }
        }

        public async Task<IEnumerable<PatientExcelHistoryModel>> FetchAllAsync(PatientExcelHistoryModel model)
        {
            var where = "where 1=1 ";

            var query = $@" Select count(UEH.*) over() as ""TotalItems"",UEH.*, A.""FullName"" as ""UploadedByName"", l.""Name"" as ""LocationName"" from ""PatientExcelHistory"" UEH 
                            join ""Account"" A on A.""AccountId"" = UEH.""UploadedBy""
                            left join ""Location"" l on l.""LocationId"" = UEH.""LocationId""
                            {where}
                            order by ""CreatedDate"" desc";

            if (model.PageIndex != null && model.PageSize != null)
            {
                model.PageIndex = model.PageIndex > 0 ? model.PageIndex - 1 : model.PageIndex;
                query += $@" limit {model.PageSize} offset {model.PageSize * model.PageIndex}";
            }

            return await this.unitOfWork.Current.QueryAsync<PatientExcelHistoryModel>(query);
        }
    }
}