﻿namespace Hims.Domain.Services
{
    using System.Collections.Generic;
    using System.Threading.Tasks;
    using Hims.Domain.Entities;
    using Hims.Shared.UserModels.Pharmacy;
    using Shared.UserModels.Vendors;

    /// <summary>
    /// The initial vendor Service interface.
    /// </summary>
    public interface IInitialVendorService
    {
        /// <summary>
        /// Inserts the product request.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<int> InsertProductRequest(List<PharmacyProductRequestModel> model);

        /// <summary>
        /// Fetches the raised products request.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<IEnumerable<PharmacyProductRequestModel>> FetchRaisedProductsRequest(PharmacyProductRequestModel model);

        /// <summary>
        /// Fetches the vendor purchase history asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<VendorPurchaseHistory> FetchVendorPurchaseHistoryAsync(PharmacyProductRequestModel model);

        /// <summary>
        /// Adds the product for approval asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<int> AddProductForApprovalAsync(VendorProductApprovalInsertModel model);

        /// <summary>
        /// Fetches the pending for final approval asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<IEnumerable<VendorProductForApprovalDisplayModel>> FetchPendingForFinalApprovalAsync(VendorProductForApprovalDisplayModel model);

        /// <summary>
        /// Fetches the sended po asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<IEnumerable<VendorViewModel>> FetchSendedPOAsync(VendorViewModel model);

        /// <summary>
        /// Fetches the vendor product details asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<IEnumerable<VendorViewDetailModel>> FetchVendorProductDetailsAsync(VendorViewDetailModel model);

        /// <summary>
        /// Fetches the approved products for vendor asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<IEnumerable<VendorViewDetailModel>> FetchApprovedProductsForVendorAsync(VendorViewDetailModel model);

        /// <summary>
        /// Adds the direct product asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<int> AddDirectProductAsync(VendorProductForApprovalModel model);

        /// <summary>
        /// Raises the po asynchronous.
        /// </summary>
        /// <param name="product">The product.</param>
        /// <returns></returns>
        Task<long> RaisePOAsync(VendorProductApprovalInsertModel product);

        /// <summary>
        /// Fetches the purchase order header asynchronous.
        /// </summary>
        /// <param name="poHeader">The po header.</param>
        /// <returns></returns>
        Task<IEnumerable<PurchaseOrderHeaderModel>> FetchPurchaseOrderHeaderAsync(PurchaseOrderHeaderModel poHeader);

        /// <summary>
        /// Fetches the purchase order detail asynchronous.
        /// </summary>
        /// <param name="poDetail">The po detail.</param>
        /// <returns></returns>
        Task<IEnumerable<PurchaseOrderDetailModel>> FetchPurchaseOrderDetailAsync(PurchaseOrderDetailModel poDetail);

        /// <summary>
        /// Fetches the only po number asynchronous.
        /// </summary>
        /// <param name="poNumber">The po number.</param>
        /// <returns></returns>
        Task<IEnumerable<PurchaseOrderHeaderModel>> FetchOnlyPONumberAsync(string poNumber);

        /// <summary>
        /// Verifies the po complete status asynchronous.
        /// </summary>
        /// <param name="purchaseBillHeaderId">The purchase bill header identifier.</param>
        /// <returns></returns>
        Task<int> VerifyPOCompleteStatusAsync(int purchaseBillHeaderId);

        /// <summary>
        /// Approves the or reject requested product.
        /// </summary>
        /// <param name="pharmacyProductRequestId">The pharmacy product request identifier.</param>
        /// <param name="isRejected">if set to <c>true</c> [is rejected].</param>
        /// <param name="createdBy">The created by.</param>
        /// <returns></returns>
        Task<int> ApproveOrRejectRequestedProduct(long pharmacyProductRequestId, bool isRejected, int createdBy);

        /// <summary>
        /// Adds the product for quotation.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<long> AddProductForQuotation(VendorProductApprovalInsertModel model);

        /// <summary>
        /// Fetches the raised product for quotation asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<IEnumerable<ProductForQuotationHeaderModel>> FetchRaisedProductForQuotationAsync(ProductForQuotationHeaderModel model);

        /// <summary>
        /// Fetches the raised product for quotation detail asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<IEnumerable<ProductForQuotationDetailModel>> FetchRaisedProductForQuotationDetailAsync(ProductForQuotationDetailModel model);

        /// <summary>
        /// Modifies the inventory request asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<int> ModifyInventoryRequestAsync(InventoryProductRequestModel model);

        /// <summary>
        /// Updates the document URL for inventory request asynchronous.
        /// </summary>
        /// <param name="request">The request.</param>
        /// <returns></returns>
        Task<int> UpdateDocumentUrlForInventoryRequestAsync(InventoryProductRequestModel request);

        /// <summary>
        /// Fetches the requested inventory product asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<IEnumerable<InventoryProductRequestModel>> FetchRequestedInventoryProductAsync(InventoryProductRequestModel model);

        /// <summary>
        /// Fetches the pending request count asynchronous.
        /// </summary>
        /// <returns></returns>
        Task<int> FetchPendingRequestCountAsync();

        /// <summary>
        /// Called when [update product in pending request asynchronous].
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<int> OnUpdateProductInPendingRequestAsync(InventoryProductRequestModel model);

        /// <summary>
        /// Called when [approve requested product asynchronous].
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<int> OnApproveRequestedProductAsync(InventoryProductRequestModel model);

        /// <summary>
        /// Called when [change supplier at final po asynchronous].
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<int> OnChangeSupplierAtFinalPOAsync(VendorProductForApprovalDisplayModel model);

        /// <summary>
        /// Fetches the purchase bill by po asynchronous.
        /// </summary>
        /// <param name="poDetail">The po detail.</param>
        /// <returns></returns>
        Task<IEnumerable<PharmacyPurchaseDetail>> FetchPurchaseBillByPOAsync(PurchaseOrderDetailModel poDetail);

        /// <summary>
        /// Fetches the purchase header asynchronous.
        /// </summary>
        /// <param name="purchaseOrderHeaderId">The purchase order header identifier.</param>
        /// <returns></returns>
        Task<PharmacyPurchaseHeader> FetchPurchaseHeaderAsync(long purchaseOrderHeaderId);

        /// <summary>
        /// Fetches the purchase bill added by GRN asynchronous.
        /// </summary>
        /// <param name="purchaseOrderHeaderId">The purchase order header identifier.</param>
        /// <returns></returns>
        Task<IEnumerable<PurchaseBillHeaderModel>> FetchPurchaseBillAddedByGrnAsync(long purchaseOrderHeaderId);
    }
}
