﻿namespace Hims.Domain.Services
{
    using System.Collections.Generic;
    using System.Threading.Tasks;
    using Shared.EntityModels;
    using Shared.UserModels.PharmacyRequest;
    using Shared.UserModels.OperationTheater;

    /// <summary>
    /// The PharmacyRequest interface.
    /// </summary>
    public interface IPharmacyRequestService
    {
        /// <summary>
        /// Gets the issued pharmacy indent detail alt asynchronous.
        /// </summary>
        /// <param name="id">The identifier.</param>
        /// <returns></returns>
        Task<IEnumerable<PharmacyIssueDisplayModel>> GetIssuedPharmacyIndentDetailAltAsync(int id);

        /// <summary>
        /// The add indent request.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        Task<int> AddIndentRequest(IndentRequestModel model);

        /// <summary>
        /// The cancel indent request.
        /// </summary>
        /// <param name="pharmacyIndentHeaderId">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        Task<int> CancelIndentRequest(int pharmacyIndentHeaderId, int pharmacyIndentDetailId);

        /// <summary>
        /// The reject indent request.
        /// </summary>
        /// <param name="indentHeaderId"></param>
        /// <param name="indentDetailId"></param>
        /// <param name="loginAccId"></param>
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        Task<int> RejectPharmacyProductIndent(int indentHeaderId, int indentDetailId, int loginAccId);

        /// <summary>
        /// The fetch pharmacy header.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        Task<IEnumerable<PharmacyIndentHeaderModel>> FetchPharmacyHeader(PharmacyIndentHeaderModel model);

        /// <summary>
        /// The fetch pharmacy detail.
        /// </summary>
        /// <param name="pharmacyIndentHeaderId">
        /// The pharmacy Indent Header Id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        Task<IEnumerable<PharmacyIndentDetailModel>> FetchPharmacyDetail(int pharmacyIndentHeaderId);

        /// <summary>
        /// The fetch all pending indent.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        Task<IEnumerable<PharmacyIndentHeaderModel>> FetchAllPendingIndent(PharmacyIndentHeaderModel model);

        /// <summary>
        /// The issue indent bill async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        Task<int> IssueIndentBillAsync(PharmacyIssueRequest model);

        /// <summary>
        /// The get issued pharmacy indent detail async.
        /// </summary>
        /// <param name="pharmacyIssueHeaderId">
        /// The pharmacyIssueHeaderId.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        Task<IEnumerable<PharmacyIssueDisplayModel>> GetIssuedPharmacyIndentDetailAsync(int pharmacyIssueHeaderId);

        /// <summary>
        /// the get product by pharmacy product
        /// </summary>
        /// <param name="pharmacyProductId"></param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        Task<string> FindProductByPharmacyProduct(int pharmacyProductId);

        /// <summary>
        /// Fetches the pharmacy detail by adm identifier.
        /// </summary>
        /// <param name="admissionId">The adm identifier.</param>
        /// <returns></returns>
        Task<IEnumerable<PharmacyIndentHeaderModel>> FetchPharmacyDetailByAdmId(int admissionId);

        /// <summary>
        /// Approves the ot indents.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<int> ApproveOTIndents(OTIssueModel model);

        /// <summary>
        /// Adds the indent return asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<int> AddIndentReturnAsync(PharmacyInPatientReturnModel model);

        /// <summary>
        /// Fetches the return request asynchronous.
        /// </summary>
        /// <param name="pharmacyIssueDetailId">The pharmacy issue detail identifier.</param>
        /// <returns></returns>
        Task<IEnumerable<PharmacyInPatientReturnModel>> FetchReturnRequestAsync(int pharmacyIssueDetailId);

        /// <summary>
        /// Fetches for return indent asynchronous.
        /// </summary>
        /// <param name="retialPharmacyId">The retial pharmacy identifier.</param>
        /// <param name="approved">if set to <c>true</c> [approved].</param>
        /// <returns></returns>
        Task<IEnumerable<PharmacyInPatientReturnModel>> FetchForReturnIndentAsync(int retialPharmacyId, bool approved);


        /// <summary>
        /// Called when [accept return asynchronous].
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<int> OnAcceptReturnAsync(PharmacyInPatientReturnModel model);

        /// <summary>
        /// Called when [reject return asynchronous].
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<int> OnRejectReturnAsync(PharmacyInPatientReturnModel model);
    }
}
