﻿namespace Hims.Domain.Services
{
    using System.Collections.Generic;
    using System.Threading.Tasks;
    using Hims.Domain.Entities.Labs;
    using Hims.Shared.UserModels.ServiceOrder;
    using Shared.EntityModels;
    using Shared.UserModels.Labs;

    /// <summary>
    /// The lab transaction service interface.
    /// </summary>
    public interface ILabTransactionService
    {
        /// <summary>
        /// Adds the lab bill asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <param name="labServicesFromServiceOrder">The lab services from service order.</param>
        /// <returns></returns>
        Task<int> AddLabBillAsync(LabBookingModel model, List<InsertLabServiceHelperModel> labServicesFromServiceOrder);

        /// <summary>
        /// Fetches the added lab bill asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<IEnumerable<LabBookingModel>> FetchAddedLabBillAsync(LabBookingModel model);


        /// <summary>
        /// Fetches the labs for sample collection asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<IEnumerable<LabSampleHandlerModel>> FetchLabsForSampleCollectionAsync(LabSampleHandlerModel model);

        /// <summary>
        /// Collects the sample and generate barcode asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<int> CollectSampleAndGenerateBarcodeAsync(LabSampleHandlerModel model, int locationId);

        /// <summary>
        /// Transfers the collected sample asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<int> TransferCollectedSampleAsync(LabTransferModel model);

        /// <summary>
        /// Fetches the transfered labs asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<IEnumerable<LabTransferModel>> FetchTransferedLabsAsync(LabTransferModel model);

        /// <summary>
        /// Fetches the lab receive sample asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<IEnumerable<LabReceiveModel>> FetchLabReceiveSampleAsync(LabReceiveModel model);

        /// <summary>
        /// Called when receive sample asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<int> OnReceiveSampleAsync(LabTransferModel model);

        /// <summary>
        /// Fetches the parameters required for input asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<LabParameterInputModel> FetchParametersRequiredForInputAsync(LabTransferModel model);

        /// <summary>
        /// Adds the parameters to labs.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<int> AddParametersToLabs(InputObservedValueModel model);

        /// <summary>
        /// Adds the template details to labs.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<int> AddTemplateDetailsToLabs(TemplateInputModel model);

        /// <summary>
        /// Fetches the lab status timeline.
        /// </summary>
        /// <param name="newLabBookingHeaderId">The new lab booking header identifier.</param>
        /// <param name="newLabBookingDetailId">The new lab booking detail identifier.</param>
        /// <returns></returns>
        Task<IEnumerable<LabBookingStatusModel>> FetchLabStatusTimeline(int? newLabBookingHeaderId, int? newLabBookingDetailId);

        /// <summary>
        /// Collects the sample and generate barcode asynchronous new.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <param name="locationId">The location identifier.</param>
        /// <returns></returns>
        Task<int> CollectSampleAndGenerateBarcodeAsyncNew(List<LabSampleHandlerModel> model, int locationId);

        /// <summary>
        /// Fetches the transfered labs asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<IEnumerable<LabTransferModel>> FetchTransferedTrackRecords(LabTransferModel model);

        /// <summary>
        /// Fetches the patient lab bill asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<IEnumerable<PatientLabBillModel>> FetchPatientLabBillAsync(PatientLabBillModel model);

        /// <summary>
        /// Uploads the lab report asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<int> UploadLabReportAsync(NewLabBookingDetailModel model);

        /// <summary>
        /// Fetches the new lab booking detail asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<NewLabBookingDetailModel> FetchNewLabBookingDetailAsync(NewLabBookingDetailModel model);

        /// <summary>
        /// Fetches the transfer header.
        /// </summary>
        /// <param name="transferHaderId">The transfer hader identifier.</param>
        /// <returns></returns>
        Task<LabTransferHeader> FetchTransferHeader(int transferHaderId);

        /// <summary>
        /// Fetches the labs for sample transfer asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<IEnumerable<LabSampleHandlerModel>> FetchLabsForSampleTransferAsync(LabSampleHandlerModel model);

        /// <summary>
        /// Fetches the new lab booking detail.
        /// </summary>
        /// <param name="NewLabBookingDetailId">The new lab booking detail identifier.</param>
        /// <returns></returns>
        Task<NewLabBookingDetail> FetchNewLabBookingDetail(int NewLabBookingDetailId);

        /// <summary>
        /// Adds the technician verification asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<int> AddTechnicianVerificationAsync(LabTechnicianVerificationModel model);

        /// <summary>
        /// Fetches the labs for technician and doctor asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<IEnumerable<LabTransferModel>> FetchLabsForTechnicianAndDoctorAsync(LabSampleHandlerModel model);

        /// <summary>
        /// Fetches the previous transfer details asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<IEnumerable<CheckTransferLocationModel>> FetchPreviousTransferDetailsAsync(CheckTransferLocationModel model);

        /// <summary>
        /// Uploads the lab consent form asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <param name="locationId">The location identifier.</param>
        /// <returns></returns>
        Task<int> UploadLabConsentFormAsync(LabSampleHandlerModel model,int locationId);

        /// <summary>
        /// Fetches the labs without cancelled asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<IEnumerable<LabSampleHandlerModel>> FetchLabsWithoutCancelledAsync(LabSampleHandlerModel model);

        /// <summary>
        /// Called when [receive sample new asynchronous].
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<int> OnReceiveSampleNewAsync(SampleReceiveNewUserModel model);

        /// <summary>
        /// Ons the department accept asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<int> onDepartmentAcceptAsync(LabTransferModel model);

        /// <summary>
        /// Transfers the collected external sample asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<int> TransferCollectedExternalSampleAsync(ExternalLabTransferUserModel model);

        /// <summary>
        /// Fetches the external transfered labs asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        Task<IEnumerable<FetchExternalLabTransferUserModel>> FetchExternalTransferedLabsAsync(FetchExternalLabTransferUserModel model);
    }
}
