﻿using Hims.Shared.UserModels.Filters;
using Hims.Shared.UserModels.Medications;

namespace Hims.Domain.Services
{
    using System.Collections.Generic;
    using System.Threading.Tasks;
    using Shared.EntityModels;
    using Shared.Library;
    using Shared.Library.Enums;
    using Shared.UserModels;
    using Hims.Shared.Library.Helpers;

    /// <summary>
    /// The Encounter Service interface.
    /// </summary>
    public interface IEncounterService
    {
        /// <summary>
        /// The get latest appointment.
        /// </summary>
        /// <param name="patientId">
        /// The patient id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        Task<BasicAppointmentModel> GetLatestAppointment(int patientId);

        /// <summary>
        /// The get uploader role.
        /// </summary>
        /// <param name="uploadedBy">
        /// The uploaded by.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        Task<UploaderModel> GetUploaderRole(int uploadedBy);

        /// <summary>
        /// The update alt async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        Task<CommonResponse> UpdateAltAsync(EncounterModel model);

        /// <summary>
        /// Gets the basic appointment details.
        /// </summary>
        /// <param name="appointmentId">The appointment identifier.</param>
        /// <param name="isAdmission">if set to <c>true</c> [is admission].</param>
        /// <returns></returns>
        Task<BasicAppointmentModel> GetBasicAppointmentDetails(int appointmentId, bool isAdmission);

        /// <summary>
        /// The get patient device tokens.
        /// </summary>
        /// <param name="appointmentId">
        /// The appointment id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        Task<IEnumerable<AccountSessionModel>> GetPatientDeviceTokens(int appointmentId);

        /// <summary>
        /// The get doctor device tokens.
        /// </summary>
        /// <param name="appointmentId">
        /// The appointment id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        Task<IEnumerable<AccountSessionModel>> GetDoctorDeviceTokens(int appointmentId);

        /// <summary>
        /// The find async.
        /// </summary>
        /// <param name="appointmentId">
        /// The encounter id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        Task<EncounterModel> FindAsync(int appointmentId);

        /// <summary>
        /// The find by appointment async.
        /// </summary>
        /// <param name="appointmentId">
        /// The appointment id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        Task<EncounterModel> FindByAppointmentAsync(int appointmentId);

        /// <summary>
        /// The add async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        Task<int> AddAsync(EncounterModel model);

        /// <summary>
        /// The update async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        Task<int> UpdateAsync(EncounterModel model);

        /// <summary>
        /// The delete async.
        /// </summary>
        /// <param name="encounterId">
        /// The encounter id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        Task<int> DeleteAsync(int encounterId);

        /// <summary>
        /// The find patient async.
        /// </summary>
        /// <param name="encounterId">
        /// The encounter id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        Task<AppointmentModel> FindPatientAsync(int encounterId);

        /// <summary>
        /// The find encounter.
        /// </summary>
        /// <param name="appointmentId">
        /// The appointment id.
        /// </param>
        /// <param name="type">
        /// The type.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        Task<EncounterResource> FindEncounterAsync(int appointmentId, InternalMedicine type);

        /// <summary>
        /// The add encounter async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        Task<int> AddEncounterAsync(EncounterModifyModel model);

        /// <summary>
        /// The add encounter alt async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        Task<CommonResponse> AddEncounterAltAsync(EncounterModifyModel model);

        /// <summary>
        /// The update encounter async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        Task<int> UpdateEncounterAsync(EncounterModifyModel model);

        /// <summary>
        /// The update encounter alt async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        Task<CommonResponse> UpdateEncounterAltAsync(EncounterModifyModel model);

        /// <summary>
        /// The find full transcript async.
        /// </summary>
        /// <param name="appointmentId">
        /// The appointment id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        Task<InternalMedicineFullTranscriptModel> FindFullTranscriptAsync(int appointmentId);
        /// <summary>
        /// The fetch active medications.
        /// </summary>
        /// <param name="patientId">
        /// The patient id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        Task<MedicationsModel> FetchActiveMedications(int patientId);

        /// <summary>
        /// The fetch vitals.
        /// </summary>
        /// <param name="model">
        /// The appointment id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        Task<IEnumerable<VitalsModel>> FetchVitals(EncounterFilterModel model);

        /// <summary>
        /// The get additional provider data async.
        /// </summary>
        /// <param name="appointmentId">
        /// The appointment id.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        Task<InternalMedicinePrescriptionModel> GetAdditionalProviderDataAsync(int appointmentId);
    }
}