﻿namespace Hims.Infrastructure.Services
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Threading.Tasks;
    using Dapper;
    using Domain.Entities;
    using Domain.Repositories.UnitOfWork;
    using Domain.Services;
    using Hims.Domain.Configurations;
    using Hims.Shared.UserModels.ProgressReport.Notes;
    using Labs = Hims.Shared.UserModels.ProgressReport.Labs;
    using MedicationInfo = Hims.Shared.UserModels.ProgressReport.MedicationInfo;

    /// <inheritdoc />
    public class ProgressReportNoteService : IProgressReportNoteService
    {
        /// <summary>
        /// The unit of work.
        /// </summary>
        private readonly IUnitOfWork unitOfWork;

        /// <summary>
        /// The amazon s3 configuration.
        /// </summary>
        private readonly IAmazonS3Configuration configuration;

        /// <inheritdoc cref="IBedService" />
        public ProgressReportNoteService(IUnitOfWork unitOfWork, IAmazonS3Configuration configuration) 
        { 
            
            this.unitOfWork = unitOfWork;
            this.configuration = configuration;
        }

        public async Task<int> DeleteAsync(DeleteModel model)
        {
            var responseId = await this.unitOfWork.ProgressReportNote.DeleteAsync(x => x.ProgressReportNoteId == model.ProgressReportNoteId);
            return responseId ? 1 : -1;
        }

        public async Task<IEnumerable<ViewModel>> FetchAsync(FilterModel model)
        {
            var dateWhere = model.Date != null
                ? $@" AND to_char(p.""Date"", 'YYYY-MM-DD' )::DATE = '{model.Date:yyyy-MM-dd}'::DATE "
                : string.Empty;

            var query = $@"SELECT
	                            p.""ProgressReportNoteId"",
	                            p.""ProgressReportId"",
	                            p.""Date""::DATE,
	                            p.""Note"",
	                            p.""CreatedDate"",
	                            c.""FullName"" ""CreatedByName"",
	                            p.""ModifiedDate"",
	                            m.""FullName"" ""ModifiedByName"",
                                pa.""Salutation"",
                                pa.""FullName"",
                                pa.""Age"",
                                pa.""Gender"",
                                pa.""UMRNo"",
                                (CASE WHEN pa.""ThumbnailUrl"" IS NOT NULL THEN CONCAT('{this.configuration.BucketURL}', pa.""Guid"", '/', pa.""ThumbnailUrl"") ELSE NULL END) AS ""PatientImage"" 
                            FROM
	                            ""ProgressReportNote""
	                            P JOIN ""ProgressReport"" r ON r.""ProgressReportId"" = P.""ProgressReportId""
                                JOIN ""Admission"" aa on aa.""AdmissionId"" = r.""AdmissionId""
	                            JOIN ""Patient"" pa on pa.""PatientId"" = aa.""PatientId""
	                            JOIN ""Account"" c on c.""AccountId"" = p.""CreatedBy""
	                            LEFT JOIN ""Account"" m on m.""AccountId"" = p.""ModifiedBy""
	                            WHERE r.""AdmissionId"" IN ({string.Join(",", model.AdmissionIds.Select(x => x).ToArray())}) {dateWhere}";
            var response = await this.unitOfWork.Current.QueryAsync<ViewModel>(query);
            return response;
        }

        public async Task<IEnumerable<MedicationInfo.LabFetchModel>> FetchNoteInfoAsync(Labs.FilterModel model)
        {
            var dateWhere = model.Date != null
                ? $@" AND to_char(p.""Date"", 'YYYY-MM-DD' )::DATE = '{model.Date:yyyy-MM-dd}'::DATE "
                : string.Empty;

            var query = $@"SELECT
	                            r.""AdmissionId"",
	                            COUNT(*) OVER() AS ""Count""
                            FROM
	                            ""ProgressReportNote""
	                            P JOIN ""ProgressReport"" r ON r.""ProgressReportId"" = P.""ProgressReportId""
	                            WHERE r.""AdmissionId"" IN ({string.Join(",", model.AdmissionIds.Select(x => x).ToArray())}) {dateWhere}";
            var response = await this.unitOfWork.Current.QueryAsync<MedicationInfo.LabFetchModel>(query);
            return response;
        }

        public async Task<int> InsertAsync(InsertModel model)
        {
            var transaction = this.unitOfWork.BeginTransaction();
            try
            {
                var progressReport = await this.unitOfWork.ProgressReport.FindAsync(x => x.AdmissionId == model.AdmissionId, transaction);
                var progressReportId = progressReport?.ProgressReportId ?? await this.unitOfWork.ProgressReport.InsertAsync(new ProgressReport
                {
                    AdmissionId = model.AdmissionId,
                    CreatedBy = model.CreatedBy,
                    CreatedDate = DateTime.Now
                }, transaction);

                var record = new ProgressReportNote
                {
                    Active = true,
                    CreatedBy = model.CreatedBy,
                    CreatedDate = DateTime.Now,
                    Date = model.Date,
                    Note = model.Note,
                    ProgressReportId = progressReportId
                };

                var responseId = await this.unitOfWork.ProgressReportNote.InsertAsync(record, transaction);
                if (responseId <= 0)
                {
                    transaction.Rollback();
                    return -1;
                }

                transaction.Commit();
                return responseId;
            }
            catch (Exception)
            {
                transaction.Rollback();
                return -1;
            }
        }

        public async Task<int> UpdateAsync(UpdateModel model)
        {
            var record = await this.unitOfWork.ProgressReportNote.FindAsync(x => x.ProgressReportNoteId == model.ProgressReportNoteId);
            record.ModifiedBy = model.ModifiedBy;
            record.ModifiedDate = DateTime.Now;
            record.Note = model.Note;

            var responseId = await this.unitOfWork.ProgressReportNote.UpdateAsync(record);
            return responseId;
        }
    }
}