﻿namespace Hims.Infrastructure.Services
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Threading.Tasks;
    using Dapper;
    using Domain.Entities;
    using Domain.Repositories.UnitOfWork;
    using Domain.Services;
    using Hims.Shared.UserModels.ProgressReport.Assessments;
    using Labs = Hims.Shared.UserModels.ProgressReport.Labs;
    using MedicationInfo = Hims.Shared.UserModels.ProgressReport.MedicationInfo;
    using Resource = Hims.Shared.UserModels.Discharge.Resource;


    /// <inheritdoc />
    public class ProgressReportAssessmentsService : IProgressReportAssessmentsService
    {
        /// <summary>
        /// The unit of work.
        /// </summary>
        private readonly IUnitOfWork unitOfWork;

        /// <inheritdoc cref="IBedService" />
        public ProgressReportAssessmentsService(IUnitOfWork unitOfWork) => this.unitOfWork = unitOfWork;

        public async Task<int> DeleteAsync(DeleteModel model)
        {
            var responseId = await this.unitOfWork.ProgressReportAssessments.DeleteAsync(x => x.ProgressReportAssessmentsId == model.ProgressReportAssessmentsId);
            return responseId ? 1 : -1;
        }

        public async Task<IEnumerable<ViewModel>> FetchAsync(FilterModel model)
        {
            var dateWhere = model.Date != null
                ? $@" AND to_char(p.""Date"", 'YYYY-MM-DD' )::DATE = '{model.Date:yyyy-MM-dd}'::DATE "
                : string.Empty;

            var query = $@"SELECT
	                            p.""ProgressReportAssessmentsId"",
	                            p.""ProgressReportId"",
	                            p.""Date""::DATE,
	                            p.""AssessmentTypeId"",
                                p.""Value"",
	                            p.""CreatedDate"",
	                            c.""FullName"" ""CreatedByName"",
	                            p.""ModifiedDate"",
	                            m.""FullName"" ""ModifiedByName"",
                                AT.""Name""
                            FROM
	                            ""ProgressReportAssessments""
	                            P join ""AssessmentType"" AT on AT.""AssessmentTypeId"" = p.""AssessmentTypeId""
                                JOIN ""ProgressReport"" r ON r.""ProgressReportId"" = P.""ProgressReportId""
	                            JOIN ""Account"" c on c.""AccountId"" = p.""CreatedBy""
	                            LEFT JOIN ""Account"" m on m.""AccountId"" = p.""ModifiedBy""
	                            WHERE r.""AdmissionId"" = {model.AdmissionId} {dateWhere} order by ""CreatedDate"" desc";
            var response = await this.unitOfWork.Current.QueryAsync<ViewModel>(query);
            return response;
        }

        public async Task<IEnumerable<MedicationInfo.LabFetchModel>> FetchAssessmentsInfoAsync(Labs.FilterModel model)
        {
            var dateWhere = model.Date != null
                ? $@" AND to_char(p.""Date"", 'YYYY-MM-DD' )::DATE = '{model.Date:yyyy-MM-dd}'::DATE "
                : string.Empty;

            var query = $@"SELECT
	                            r.""AdmissionId"",
	                            COUNT(*) OVER() AS ""Count""
                            FROM
	                            ""ProgressReportAssessments""
	                            P JOIN ""ProgressReport"" r ON r.""ProgressReportId"" = P.""ProgressReportId""
	                            WHERE r.""AdmissionId"" IN ({string.Join(",", model.AdmissionIds.Select(x => x).ToArray())}) {dateWhere}";
            var response = await this.unitOfWork.Current.QueryAsync<MedicationInfo.LabFetchModel>(query);
            return response;
        }

        public async Task<int> InsertAsync(InsertModel model)
        {
            var transaction = this.unitOfWork.BeginTransaction();
            try
            {
                var responseId = 0;
                var progressReport = await this.unitOfWork.ProgressReport.FindAsync(x => x.AdmissionId == model.AdmissionId, transaction);
                var progressReportId = progressReport?.ProgressReportId ?? await this.unitOfWork.ProgressReport.InsertAsync(new ProgressReport
                {
                    AdmissionId = model.AdmissionId,
                    CreatedBy = model.CreatedBy,
                    CreatedDate = DateTime.Now
                }, transaction);
                for (int i = 0; i < model.Assessments.Count; i++)
                {
                    var record = new ProgressReportAssessments
                    {
                        Active = true,
                        CreatedBy = model.CreatedBy,
                        CreatedDate = DateTime.Now,
                        Date = model.Date,
                        AssessmentTypeId = model.Assessments[i].Id,
                        Value = model.Assessments[i].Value,
                        ProgressReportId = progressReportId
                    };

                    responseId = await this.unitOfWork.ProgressReportAssessments.InsertAsync(record, transaction);
                    if (responseId <= 0)
                    {
                        transaction.Rollback();
                        return -1;
                    }
                }

                transaction.Commit();
                return responseId;
            }
            catch (Exception)
            {
                transaction.Rollback();
                return -1;
            }
        }

        public async Task<int> UpdateAsync(UpdateModel model)
        {
            var record = await this.unitOfWork.ProgressReportAssessments.FindAsync(x => x.ProgressReportAssessmentsId == model.Assessments[0].ProgressReportAssessmentsId);
            record.AssessmentTypeId = model.Assessments[0].Id;
            record.Value = model.Assessments[0].Value;
            record.ProgressReportAssessmentsId = model.Assessments[0].ProgressReportAssessmentsId;
            record.ProgressReportId = model.Assessments[0].ProgressReportId;
            record.ModifiedBy = model.Assessments[0].ModifiedBy;
            record.ModifiedDate = DateTime.Now;
            record.Date = model.Date;
            record.Active = true;        
               
            int responseId = await unitOfWork.ProgressReportAssessments.UpdateAsync(record);          
            return responseId;
        }

        public async Task<IEnumerable<Resource.ViewModel>> FetchAssessmentTypesAsync()
        {
            var query = $@"SELECT
	                        ""AssessmentTypeId"" AS ""Id"",
	                        ""Name"" AS ""Name"" 
                        FROM
	                        ""AssessmentType"" ";
            var records = await this.unitOfWork.Current.QueryAsync<Resource.ViewModel>(query);
            return records;
        }
    }
}